﻿<# シリアル通信スクリプト 2023/11/06
・シリアル通信および、AP機器とのコマンドとステータスの処理をするスクリプトです。
・AP機器にあわせた通信設定とコマンド指定を行ってください。
#>
################################################## パラメータ
$Active_FileName = "ApTool_実行中.txt"  #スクリプト実行中ファイル。このファイルを削除することでスクリプトを終了できます。
$ComLog_FileName = "ComLog.txt"         #シリアル通信ログファイル。不要な場合は""を指定
$ErrStat_Path = ""       #エラーステータスファイル。不要な場合は""を指定

# シリアル通信パラメータ設定
$ComParam = "COM2",38400,"None",8,"One"  #COMポート番号,ボーレート,パリティ,データ長,ストップビット
$Handshake = "None"   #ハンドシェイク設定 None、XOnXOff、RequestToSend、RequestToSendXOnXOff
$RecvTimeout = 20     #受信タイムアウト時間(ms) 0で末尾CRまで無限待ち ※通常は変更不要ですが、受信データが途切れる場合などに変更してください。
$TerminateChr = "`r"  #受信データの終端文字 ""でタイムアウトするまで受信 ※通常は変更不要

#AP機器コマンド設定（機種毎に設定）改行コードCRは"`r"を指定
#$Cmd = ",,,,"                       #AP-1207
#$Cmd = "I,,F,F,R"                   #AP-1500
$Cmd = "I,,A,P0`r,P1`r"             #AP-2000
#$Cmd = "I,,A,P000`r,P100`r"         #AP-2200
#$Cmd = "I,L050,F,P000`r,P100`r"     #AP-2200 読取位置を変更する場合
#$Cmd = "I`r,,A`r,P00`r,P01`r"       #AP-3100 QR読取
#$Cmd = "I`r,,AU0010`r,P00`r,P01`r"  #AP-3100 RFID読取
#$Cmd = "I,,A,P00`r,P01`r"          #AP-3200
#$Cmd = "I,,A,P00`r,P01`r"          #AP-3600 かんばん 
#$Cmd = "I`r,,F`r,P0`r,P1`r"         #AP-9300
#$Cmd = "I`r,B0100`r,F`r,P0`r,P1`r"    #AP-9500 QR

#以下の項目は変更不要です。
$CmdAry = $Cmd.Split(",")   #カンマ区切りで配列へ代入
$CommI = $CmdAry[0]; $CommB = $CmdAry[1]; $CommF = $CmdAry[2]; $CommP0 = $CmdAry[3]; $CommP1 = $CmdAry[4]
#  $CommI  装置リセットコマンド。"I"または"I`r"
#  $CommB  動作設定コマンド。Bコマンド、Lコマンド等の読取命令前に送るコマンド。
#  $CommF  読取開始コマンド。自動読取は"A"にする。RFID読取時は"F0010"などにする。
#  $CommP0 読取OK時の排出コマンド。機種により"P0"などにする。続けて印字文字を指定可能です。
#  $CommP1 読取エラー時の排出コマンド。機種により"P1"などにする。続けて印字文字を指定可能です。

################################################## シリアル通信COMポート設定・オープン
# COMポート生成、パラメータ設定
$ComPortObj = New-Object System.IO.Ports.SerialPort $ComParam    #COM番号、ボーレート、データ長、パリティ、ストップ
$ComPortObj.DtrEnable = $true        #DTR設定
$ComPortObj.RtsEnable = $true        #RTS設定
$ComPortObj.Handshake = $Handshake   #ハンドシェイク設定 (None、XOnXOff、RequestToSend、RequestToSendXOnXOff)
#$ComPortObj.NewLine = $TerminateChr  #改行文字設定 デフォルト"`n"（WriteLineやReadLineメソッドに適用される）
$ComPortObj.Encoding=[System.Text.Encoding]::GetEncoding("Shift_JIS")    # 文字コード設定

# シリアルポートエラーイベント（エラー発生時に$trueを返す）通信でパリティエラーが発生した場合など
$ComErrEventObj = Register-ObjectEvent -InputObject $ComPortObj -EventName "ErrorReceived" -Action {$true}

#---+---+---+---+---+---+---+---+---+---+ COMポートのオープン
Try {
    $ComPortObj.Open()
} Catch {
    [void]$oWS.Popup("シリアルCOMポート（" + $ComParam[0] +"）のオープンが出来ません。`n読取を終了します。", 0, $WinTitle, 0 + 16)
    If (Test-Path $Active_FileName) { Remove-Item -Path $Active_FileName }    #実行中ファイル削除
    Exit  #スクリプト終了
}

# バッファデータクリア
Start-Sleep -s 1  #ポートオープン時の不要データ削除用に待機
$ComPortObj.DiscardInBuffer()    #受信バッファーのクリア
$ComPortObj.DiscardOutBuffer()   #送信バッファーのクリア

################################################## ファイル作成
Set-Location -Path $PSScriptRoot    #カレントディレクトリをスクリプトのディレクトリへ変更
New-Item $Active_FileName -Force    #実行中ファイル作成
If ($ComLog_FileName -ne "") {
    New-Item $ComLog_FileName -Force    #通信ログファイル作成 新規上書き
}

################################################## シリアル通信 関数定義
#---+---+---+---+---+---+---+---+---+---+ データ送信
Function SendSub($SendStr) {
    CtsDsrChk    #機器接続チェック
    $ComPortObj.Write($SendStr)    #送信

    If ($ComLog_FileName -ne "") {
        $now = Get-Date -Format "yyyy/MM/dd HH:mm:ss"
        Write-Output "$now 送> $SendStr" | Out-File -FilePath $ComLog_FileName -Encoding Default -Append  #通信データをファイルへ追記
    }

    #通信エラー確認
    If (Receive-job -job $ComErrEventObj) {
        [void]$oWS.Popup("送信エラーが発生しました。機器の電源を入れ直してください。`n$SendStr", 0, $WinTitle, 0 + 16)
    }
}

#---+---+---+---+---+---+---+---+---+---+ データ受信
#戻り値：ステータスコードまたは、読取データ  終了またはエラー時は""
Function RecvSub {
    CtsDsrChk    #機器接続チェック

    #受信待ち
    While ($ComPortObj.BytesToRead -eq 0) {    #受信データ有無確認
        If (-not (Test-Path $Active_FileName)) { Return "" }   #実行中ファイルが存在しなければ終了
        Start-Sleep -m 1    #受信待ち 1ms
    }

    $RecvStr = ""
    $Cnt = 0
    Do {
        If ($ComPortObj.BytesToRead -gt 0) {    #受信データ有無確認
            $RecvStr += $ComPortObj.ReadExisting()    #受信
            $Cnt = 0
        } Else {
            If ($TerminateChr -ne "") {
                If ($RecvStr -match "$TerminateChr$") { Break }    #データ末尾が終端文字なら受信終了
            }
            If ($RecvTimeout -gt 0) { $Cnt++ }    #受信待ちカウントアップ
            Start-Sleep -m 1    #受信待ち 1ms
        }
    } While ($Cnt -le $RecvTimeout)    #受信タイムアウト確認


    If ($ComLog_FileName -ne "") {
        $now = Get-Date -Format "yyyy/MM/dd HH:mm:ss"
        Write-Output "$now 受< $RecvStr" | Out-File -FilePath $ComLog_FileName -Encoding Default -Append  #通信データをファイルへ追記
    }

    #終端文字確認
    If ($TerminateChr -ne "") {
        If ($RecvStr -notmatch "$TerminateChr$") {
            [void]$oWS.Popup("データの終端文字が受信できませんでした。`n$RecvStr", 0, $WinTitle, 0 + 16)
        }
    }

    #通信エラー確認
    If (Receive-job -job $ComErrEventObj) {
        [void]$oWS.Popup("受信エラーが発生しました。機器の電源を入れ直してください。`n$RecvStr", 0, $WinTitle, 0 + 16)
        Return ""
    }

    $RecvStr = $RecvStr.Substring(0, $RecvStr.Length - 1)    #末尾CRを削除
    Return $RecvStr
}

#---+---+---+---+---+---+---+---+---+---+ シリアル通信の制御線CTS/DSR確認
Function CtsDsrChk {
    While (-not ($ComPortObj.CtsHolding -or $ComPortObj.DsrHolding)) {
        If ($oWS.Popup("機器の電源および接続を確認してください。`nキャンセルで読取を終了します。", 0, $WinTitle, 5 + 16) -eq 2) {  #キャンセルボタン確認
            If (Test-Path $Active_FileName) {
                Remove-Item -Path $Active_FileName  #実行中ファイル削除
            }
            Break
        }
    }
}

################################################## スタート時コマンド処理
Function StartSub($Comm) {
    If (ResetSub $false) {
        If ($CommB -ne "") {
            SendSub "$CommB"    #動作設定コマンド
            $StatStr = RecvSub    #ステータス受信
            If ($StatStr -ne "$([char]0x1b)0") {     #ステータスが正常応答か確認
                MainSub $StatStr
                Return
            }
        }
        SendSub $CommF    #読取開始コマンド
    }
}

################################################## リセット時コマンド処理
#引数：trueで読取開始コマンドを続けて送信する
#戻り値：エラー時はfalse
Function ResetSub($Flag) {
    SendSub $CommI    #リセットコマンド
    If ($CommI -match "`r$") {
        $StatStr = RecvSub    #ステータス受信
        If ($StatStr -ne "$([char]0x1b)0") {     #ステータスが正常応答か確認
            If ($StatStr -ne "") { [void]$oWS.Popup("装置リセットでエラーが発生しました。機器の電源を入れ直してください。", 0, $WinTitle, 0 + 16) }
            Return $false
        }
    }
    If ($Flag) { SendSub $CommF.Replace("A", "F") }    #スタートキー待ちのためFコマンド変更して送信
    Return $true
}

################################################## メインループ処理
Function MainSub() {
    $StatStr = RecvSub  #ステータス受信

    If ($StatStr.Length -ge 2) {    #データが2バイト以上か確認
        If ($StatStr -match "$([char]0x1b)P$") {    #電源投入ESC+Pは末尾2バイトを確認（電源投入時に受信されるノイズデータ対策）
            ErrStatSub $StatStr "機器の電源投入" 64
            StartSub
        } ElseIf ($StatStr.Substring(0, 1) -eq [char]0x1b) {    #先頭ESC確認
            Switch ($StatStr.Substring(1, 1)) {
                "0" {    #正常応答
                    SendSub $CommF
                }
                "?" {    #読取エラー
                    ReadNgSub
                }
                "J" {    #走行エラー
                    ErrStatSub $StatStr "走行エラー" 16
                    $null = ResetSub $true
                }
                "U" {    #走行エラー
                    ErrStatSub $StatStr "走行エラー" 16
                    $null = ResetSub $true
                }
                "D" {    #ダブルフィードエラー
                    ErrStatSub $StatStr "ダブルフィードエラー" 16
                    $null = ResetSub $true
                }
                "*" {    #排出エラー
                    ErrStatSub $StatStr "排出エラー" 16
                    $null = ResetSub $true
                }
                "C" {    #コマンドエラー
                    ErrStatSub $StatStr "コマンドエラー" 16
                    $null = ResetSub $true
                }
                "H" {    #ホッパーエンプティ
                    ErrStatSub $StatStr "ホッパーエンプティ" 64
                    SendSub $CommF.Replace("A", "F")  #Fコマンドに変更
                }
                "E" {    #終了ボタン
                    If ($oWS.Popup("終了キーが押されました。読取を終了しますか？", 0, $WinTitle, 1 + 32) -eq 1) {
                        Remove-Item -Path $Active_FileName    #実行中ファイル削除 スクリプト終了
                    } Else {
                        StartSub
                    }
                }
                Default {    #未対応ステータス
                    ErrStatSub $StatStr "未対応ステータス" 16
                    $null = ResetSub $true
                }
            }
        } Else {    #読取OK
            ReadOkSub $StatStr
        }
    } ElseIf ($StatStr -ne "") {
        ReadOkSub $StatStr    #読取データ1バイト
    }
}

################################################## エラーステータス処理
Function ErrStatSub($Stat, $Msg, $Type) {
    [void]$oWS.Popup($Msg, 2, $WinTitle, $Type)  #ポップアップ表示 2秒間
    $Stat = $Stat.Substring(1, $Stat.Length - 1)  #先頭ESCを削除
    Write-Host "$Msg（$Stat）" -ForegroundColor Yellow  #エラーステータスをコンソール表示

    #エラーステータスファイル作成
    If ($ErrStat_Path -ne "") {
        New-Item $ErrStat_Path -Force -Value $Stat  #ステータスファイル作成 上書き
    }
}

################################################## シリアル通信COMポートのクローズ
Function ComCloseSub() {
    If ($ComPortObj.IsOpen) {  #COMポートがオープンされているか確認
        $null = ResetSub $false  #機器リセット
        $ComPortObj.Close()  # COMポートのクローズ
    }
    If (Test-Path $Active_FileName) {
        Remove-Item -Path $Active_FileName  #実行中ファイルが残っていれば削除
    }
}